/* A Simple Abstract Shmup Game
 Copyright (C) 2020 Florent Monnier
 
 This software is provided "AS-IS", without any express or implied warranty.
 In no event will the authors be held liable for any damages arising from
 the use of this software.
 
 Permission is granted to anyone to use this software for any purpose,
 including commercial applications, and to alter it and redistribute it freely.
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <SDL.h>

unsigned int width = 640;
unsigned int height = 480;

unsigned int shot = 0;
unsigned int missed = 0;

#define bool int
#define true 1
#define false 0

struct point2d {
    int x;
    int y;
};

struct rgb {
    Uint8 r;
    Uint8 g;
    Uint8 b;
};

/* Timeline types */

struct timeline_chunk_from {
    int time;
    struct point2d p;
    /* after time [t] is reach (and before next timeline chunk)
       the returned value will be [p] */
};

struct timeline_chunk_evol {
    int time1;
    int time2;
    struct point2d qbcurve[3];
    /* when [t] is between [t1] and [t2] the value is
       the result of [f t1 t2 t d] */
};

#define TIMELINE_TYPE_FROM 0
#define TIMELINE_TYPE_EVOL 1

struct timeline {
    int type;
    union {
        struct timeline_chunk_from from;
        struct timeline_chunk_evol evol;
    };
};


struct foe {
    struct point2d foe_pos;
    struct timeline foe_anim[3];
    int foe_anim_len;
    int foe_last_shot;
    int foe_shoot_freq;
    SDL_Texture *foe_texture;
};


struct foe_bullet {
    struct point2d bullet_pos;
    struct point2d bullet_line[2];
    int bullet_birth;
};


struct player_dir {
    bool left;
    bool right;
    bool up;
    bool down;
};

struct player {
    struct point2d p_pos;
    int p_last_shot;
    int p_shoot_freq;
    bool p_shooting;
    struct player_dir p_dir;
    SDL_Texture *p_texture;
};


#define MAX_NUM_FOES 30

#define MAX_F_BULLETS_NUM 300
#define MAX_P_BULLETS_NUM 200

struct game_state {
  struct player player;

  struct foe foes[MAX_NUM_FOES];
  int foes_num;

  struct foe_bullet f_bullets[MAX_F_BULLETS_NUM];  /* foes bullets */
  int f_bullets_num;

  struct point2d p_bullets[MAX_P_BULLETS_NUM];  /* player bullets */
  int p_bullets_num;

  bool game_over;
};

struct game_data {
    SDL_Renderer *renderer;
    SDL_Texture *f_bullet_tex;
    SDL_Texture *p_bullet_tex;
    SDL_Texture *letters_tex[256];
};

Uint8 background[12][16];

struct letter_pattern {
    unsigned char letter;
    char pattern[5][5];
};

struct letter_pattern letters[] = {
    { '0', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 1, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { '1', {
      { 0, 0, 1, 0, 0 },
      { 0, 1, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 1, 1, 1, 0 },
    }},
    { '2', {
      { 1, 1, 1, 0, 0 },
      { 0, 0, 0, 1, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 1, 0, 0, 0 },
      { 1, 1, 1, 1, 0 },
    }},
    { '3', {
      { 1, 1, 1, 0, 0 },
      { 0, 0, 0, 1, 0 },
      { 0, 1, 1, 0, 0 },
      { 0, 0, 0, 1, 0 },
      { 1, 1, 1, 0, 0 },
    }},
    { '4', {
      { 0, 0, 0, 1, 0 },
      { 0, 0, 1, 1, 0 },
      { 0, 1, 0, 1, 0 },
      { 1, 1, 1, 1, 1 },
      { 0, 0, 0, 1, 0 },
    }},
    { '5', {
      { 1, 1, 1, 1, 1 },
      { 1, 0, 0, 0, 0 },
      { 1, 1, 1, 1, 0 },
      { 0, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
    }},
    { '6', {
      { 0, 0, 1, 1, 0 },
      { 0, 1, 0, 0, 0 },
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { '7', {
      { 1, 1, 1, 1, 1 },
      { 0, 0, 0, 1, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 1, 0, 0, 0 },
      { 1, 0, 0, 0, 0 },
    }},
    { '8', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { '9', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 1 },
      { 0, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { ' ', {
      { 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0 },
    }},
    { ':', {
      { 0, 0, 0, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 0, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 0, 0, 0 },
    }},
    { '-', {
      { 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0 },
      { 0, 1, 1, 1, 0 },
      { 0, 0, 0, 0, 0 },
      { 0, 0, 0, 0, 0 },
    }},
    { 'a', {
      { 0, 0, 1, 0, 0 },
      { 0, 1, 0, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 1 },
      { 1, 0, 0, 0, 1 },
    }},
    { 'b', {
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
    }},
    { 'c', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { 'd', {
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
    }},
    { 'e', {
      { 1, 1, 1, 1, 1 },
      { 1, 0, 0, 0, 0 },
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 1, 1, 1, 1 },
    }},
    { 'f', {
      { 1, 1, 1, 1, 1 },
      { 1, 0, 0, 0, 0 },
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 0, 0 },
    }},
    { 'g', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 1, 1 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { 'h', {
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
    }},
    { 'i', {
      { 0, 1, 1, 1, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 1, 1, 1, 0 },
    }},
    { 'j', {
      { 0, 0, 1, 1, 1 },
      { 0, 0, 0, 1, 0 },
      { 0, 0, 0, 1, 0 },
      { 1, 0, 0, 1, 0 },
      { 0, 1, 1, 0, 0 },
    }},
    { 'k', {
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 1, 0 },
      { 1, 1, 1, 0, 0 },
      { 1, 0, 0, 1, 0 },
      { 1, 0, 0, 0, 1 },
    }},
    { 'l', {
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 1, 1, 1, 0 },
    }},
    { 'm', {
      { 1, 0, 0, 0, 1 },
      { 1, 1, 0, 1, 1 },
      { 1, 0, 1, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
    }},
    { 'n', {
      { 1, 0, 0, 0, 1 },
      { 1, 1, 0, 0, 1 },
      { 1, 0, 1, 0, 1 },
      { 1, 0, 0, 1, 1 },
      { 1, 0, 0, 0, 1 },
    }},
    { 'o', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { 'p', {
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 0 },
      { 1, 0, 0, 0, 0 },
    }},
    { 'q', {
      { 0, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 1, 0 },
      { 0, 1, 1, 0, 1 },
    }},
    { 'r', {
      { 1, 1, 1, 1, 0 },
      { 1, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
      { 1, 0, 1, 0, 0 },
      { 1, 0, 0, 1, 0 },
    }},
    { 's', {
      { 0, 1, 1, 1, 1 },
      { 1, 0, 0, 0, 0 },
      { 0, 1, 1, 1, 0 },
      { 0, 0, 0, 0, 1 },
      { 1, 1, 1, 1, 0 },
    }},
    { 't', {
      { 1, 1, 1, 1, 1 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
    }},
    { 'u', {
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 1, 1, 0 },
    }},
    { 'v', {
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 0, 1, 0, 1, 0 },
      { 0, 0, 1, 0, 0 },
    }},
    { 'w', {
      { 1, 0, 0, 0, 1 },
      { 1, 0, 0, 0, 1 },
      { 1, 0, 1, 0, 1 },
      { 1, 0, 1, 0, 1 },
      { 0, 1, 0, 1, 0 },
    }},
    { 'x', {
      { 1, 0, 0, 0, 1 },
      { 0, 1, 0, 1, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 1, 0, 1, 0 },
      { 1, 0, 0, 0, 1 },
    }},
    { 'y', {
      { 1, 0, 0, 0, 1 },
      { 0, 1, 0, 1, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 0, 1, 0, 0 },
    }},
    { 'z', {
      { 1, 1, 1, 1, 1 },
      { 0, 0, 0, 1, 0 },
      { 0, 0, 1, 0, 0 },
      { 0, 1, 0, 0, 0 },
      { 1, 1, 1, 1, 1 },
    }},
};


void p2_add(
        struct point2d a,
        struct point2d b,
        struct point2d *r) {
    r->x = a.x + b.x;
    r->y = a.y + b.y;
}

void p2_sub(
        struct point2d a,
        struct point2d b,
        struct point2d *r) {
    r->x = a.x - b.x;
    r->y = a.y - b.y;
}

void p2_mul(
        struct point2d p,
        int k,
        struct point2d *r) {
    r->x = p.x * k;
    r->y = p.y * k;
}

void p2_div(
        struct point2d p,
        int k,
        struct point2d *r) {
    r->x = p.x / k;
    r->y = p.y / k;
}



/* returns a point on the quadratic bezier curve defined by p1, p2 and p3,
   with t in the interval for interpolation (0, 1000) instead of (0.0, 1.0)
   for floats. */
void QuadraticBezierCurve(
        struct point2d p1,
        struct point2d p2,
        struct point2d p3,
        int t,
        struct point2d *r)
{
    struct point2d v1;
    struct point2d v2;
    struct point2d v3;
    struct point2d v4;
    struct point2d v5;

    int ti = 1000 - t;

    /*
        ( p1 * ((ti * ti) / 1000) +
          p2 * ((2 * ti * t) / 1000) +
          p3 * ((t * t) / 1000)
        ) / 1000
    */

    p2_mul(p1, ((ti * ti) / 1000), &v1);
    p2_mul(p2, ((2 * ti * t) / 1000), &v2);
    p2_mul(p3, ((t * t) / 1000), &v3);

    p2_add(v1, v2, &v4);
    p2_add(v3, v4, &v5);

    p2_div(v5, 1000, r);
}

int inter1(int t, int t1, int t2, int v1, int v2)
{
    return (((v2 - v1) * (t - t1)) / (t2 - t1) + v1);
}

void point_on_qbcurve(
        int t1, int t2, int t,
        struct point2d p1,
        struct point2d p2,
        struct point2d p3,
        struct point2d *r)
{
    int _t = inter1(t, t1, t2, 0, 1000);
    QuadraticBezierCurve(p1, p2, p3, _t, r);
}


void return_point(struct point2d p, struct point2d *r)
{
    r->x = p.x;
    r->y = p.y;
}


void timeline_val_at(struct timeline *tl, int len, int t, struct point2d *r)
{
    int i, j;
    for (i = 0; i < (len - 1); i++) {
        j = i + 1;
        if (tl[i].type == TIMELINE_TYPE_FROM &&
            tl[j].type == TIMELINE_TYPE_FROM )
        {
            int t1 = tl[i].from.time;
            int t2 = tl[j].from.time;
            if (t1 <= t && t < t2) {
                return_point(tl[i].from.p, r);
                return ;
            }
        }
        if (tl[i].type == TIMELINE_TYPE_FROM &&
            tl[j].type == TIMELINE_TYPE_EVOL )
        {
            int t1 = tl[i].from.time;
            int t2 = tl[j].evol.time1;
            if (t1 <= t && t < t2) {
                return_point(tl[i].from.p, r);
                return ;
            }
        }
        if (tl[i].type == TIMELINE_TYPE_EVOL)
        {
            int t1 = tl[i].evol.time1;
            int t2 = tl[i].evol.time2;
            if (t1 <= t && t <= t2)
            {
                struct point2d _r;
                point_on_qbcurve(
                        t1, t2, t,
                        tl[i].evol.qbcurve[0],
                        tl[i].evol.qbcurve[1],
                        tl[i].evol.qbcurve[2],
                        &_r);
                return_point(_r, r);
                return ;
            }
        }
    }

    i = len - 1;
    if (tl[i].type == TIMELINE_TYPE_FROM) {
        return_point(tl[i].from.p, r);
        return ;
    }

    if (tl[i].type == TIMELINE_TYPE_EVOL)
    {
        int t1 = tl[i].evol.time1;
        int t2 = tl[i].evol.time2;

        if (t1 <= t && t <= t2)
        {
            struct point2d _r;
            point_on_qbcurve(
                    t1, t2, t,
                    tl[i].evol.qbcurve[0],
                    tl[i].evol.qbcurve[1],
                    tl[i].evol.qbcurve[2],
                    &_r);
            return_point(_r, r);
        } else
        if (t >= t2)
        {
            struct point2d _r;
            point_on_qbcurve(
                    t1, t2, t2,
                    tl[i].evol.qbcurve[0],
                    tl[i].evol.qbcurve[1],
                    tl[i].evol.qbcurve[2],
                    &_r);
            return_point(_r, r);
        }
    }
}


int timeline_finished(struct timeline *tl, int len, int t)
{
    int i;
    for (i = 0; i < (len - 1); i++) {
        if (tl[i].type == TIMELINE_TYPE_FROM) {
            int t2 = tl[i].from.time;
            if (t < t2) return 0;
        }
        if (tl[i].type == TIMELINE_TYPE_EVOL) {
            int t2 = tl[i].evol.time2;
            if (t < t2) return 0;
        }
    }
    i = len - 1;
    if (tl[i].type == TIMELINE_TYPE_FROM) {
        return 1;
    }
    if (tl[i].type == TIMELINE_TYPE_EVOL) {
        int t2 = tl[i].evol.time2;
        return (t > t2);
    }
}


void timeline_test_at(struct timeline *tl, int len, int t)
{
    struct point2d r;
    timeline_val_at(tl, len, t, &r);
    printf("t(%d): (%d, %d)\n", t, r.x, r.y);
}

void test_timeline()
{
    struct timeline tl[10];
    int len = 5;

    tl[0].type = TIMELINE_TYPE_FROM;
    tl[0].from.time = 0;
    tl[0].from.p.x = 0;
    tl[0].from.p.y = 0;

    tl[1].type = TIMELINE_TYPE_FROM;
    tl[1].from.time = 3;
    tl[1].from.p.x = 20;
    tl[1].from.p.y = 20;

    tl[2].type = TIMELINE_TYPE_FROM;
    tl[2].from.time = 5;
    tl[2].from.p.x = 40;
    tl[2].from.p.y = 40;

    tl[3].type = TIMELINE_TYPE_EVOL;
    tl[3].evol.time1 = 6;
    tl[3].evol.time2 = 9;
    tl[3].evol.qbcurve[0].x = 0;
    tl[3].evol.qbcurve[0].y = 0;
    tl[3].evol.qbcurve[1].x = 500;
    tl[3].evol.qbcurve[1].y = 0;
    tl[3].evol.qbcurve[2].x = 500;
    tl[3].evol.qbcurve[2].y = 400;

    tl[4].type = TIMELINE_TYPE_FROM;
    tl[4].from.time = 9;
    tl[4].from.p.x = 100;
    tl[4].from.p.y = 100;

    timeline_test_at(tl, len, 1);
    timeline_test_at(tl, len, 4);
    timeline_test_at(tl, len, 5);

    timeline_test_at(tl, len, 6);
    timeline_test_at(tl, len, 7);
    timeline_test_at(tl, len, 8);
    timeline_test_at(tl, len, 9);
    timeline_test_at(tl, len, 10);
}


struct pausable_timer {
    Uint32 t1;
    Uint32 time;
    bool paused;
};

struct pausable_timer ptimer;

void init_pausable_timer(struct pausable_timer *pt)
{
    pt->t1 = 0;
    pt->time = 0;
    pt->paused = false;
}

bool timer_is_paused(struct pausable_timer *pt)
{
    return pt->paused;
}

Uint32 timer_get_ticks(struct pausable_timer *pt)
{
    if (pt->paused) {
        return (pt->t1 - pt->time);
    } else {
        Uint32 t = SDL_GetTicks();
        return (t - pt->time);
    }
}

void timer_pause(struct pausable_timer *pt)
{
    if (!pt->paused) {
        pt->t1 = SDL_GetTicks();
        pt->paused = true;
    }
}

void timer_restart(struct pausable_timer *pt)
{
    if (pt->paused) {
        Uint32 t2 = SDL_GetTicks();
        pt->time = pt->time + (t2 - pt->t1);
        pt->paused = false;
    }
}

void timer_toggle_pause(struct pausable_timer *pt)
{
    if (!pt->paused)
        timer_pause(pt);
    else
        timer_restart(pt);
}


void SDL_RenderFillRect_(
        SDL_Renderer* renderer,
        const SDL_Rect* rect)
{
    int err = SDL_RenderFillRect(renderer, rect);
    if (err != 0) {
        fprintf(stderr, "SDL_RenderFillRect() failed: %s\n", SDL_GetError());
    }
}

void SDL_SetRenderDrawColor_(
        SDL_Renderer* renderer,
        Uint8 r, Uint8 g, Uint8 b, Uint8 a)
{
    int err = SDL_SetRenderDrawColor(renderer, r, g, b, a);
    if (err != 0) {
        fprintf(stderr, "SDL_SetRenderDrawColor() failed: %s\n", SDL_GetError());
    }
}

int rand_int(int n)
{
   return (int)(rand() / (double)RAND_MAX * (n - 1));
}

int rand_bool()
{
   return (rand() < (RAND_MAX / 2));
}


void fill_rect40(SDL_Renderer *renderer, Uint8 r, Uint8 g, Uint8 b, int x, int y)
{
    SDL_Rect rect = {
        .x = x,
        .y = y,
        .w = 40,
        .h = 40
    };
    SDL_SetRenderDrawColor_(renderer, r, g, b, 255);
    int _r = SDL_RenderFillRect(renderer, &rect);
}


void init_background()
{
    int x, y;
    for (y = 0; y < 12; y++) {
        for (x = 0; x < 16; x++) {
            background[y][x] = 90 + rand_int(20);
        }
    }
}


void step_background()
{
    int x, y;
    for (y = 0; y < 12; y++) {
        for (x = 0; x < 16; x++) {
            if (rand_int(8000) < 2)
                background[y][x] = 90 + rand_int(20);
        }
    }
}


void display_background(SDL_Renderer *renderer, bool game_over)
{
    int x, y;
    for (y = 0; y < 12; y++) {
        for (x = 0; x < 16; x++) {
            int _x = x * 40;
            int _y = y * 40;
            Uint8 v = background[y][x];
            if (!game_over) {
                fill_rect40(renderer, v, v, v, _x, _y);
            } else {
                fill_rect40(renderer,
                    (v + 40),
                    (v / 2),
                    (v / 3),  _x, _y);
            }
        }
    }
}


void display(struct game_state *game_state, struct game_data *game_data)
{
    SDL_Rect src_rect = { .x = 0, .y = 0, .w = 5, .h = 5 };
    int i, j;

    display_background(game_data->renderer, game_state->game_over);

    /* display infos */
    {
        unsigned char s[60];
        unsigned char c;
        int r, i, len;
        SDL_Rect dst_rect = {
            .x = 0,
            .y = 0,
            .w = 20,
            .h = 20
        };
        sprintf(s, "shot: %d", shot);
        len = strlen(s);
        for (i = 0; i < len; i++) {
            c = s[i];
            dst_rect.x = i * 20 + 10;
            dst_rect.y = 10;
            dst_rect.w = 15;
            dst_rect.h = 15;
            r = SDL_RenderCopy(
                game_data->renderer,
                game_data->letters_tex[c],
                &src_rect, &dst_rect);
        }
        sprintf(s, "missed: %d", missed);
        len = strlen(s);
        for (i = 0; i < len; i++) {
            c = s[i];
            dst_rect.x = i * 15 + width - 170;
            dst_rect.y = 10;
            dst_rect.w = 10;
            dst_rect.h = 10;
            r = SDL_RenderCopy(
                game_data->renderer,
                game_data->letters_tex[c],
                &src_rect, &dst_rect);
        }
        sprintf(s, "score: %d", (shot - missed));
        len = strlen(s);
        for (i = 0; i < len; i++) {
            c = s[i];
            dst_rect.x = i * 15 + 10;
            dst_rect.y = height - 25;
            dst_rect.w = 10;
            dst_rect.h = 10;
            r = SDL_RenderCopy(
                game_data->renderer,
                game_data->letters_tex[c],
                &src_rect, &dst_rect);
        }
    }

    /* display foes' bullets */
    for (i = 0; i < game_state->f_bullets_num; i++)
    {
        int r;
        struct foe_bullet *f_bullet = &game_state->f_bullets[i];
        SDL_Rect dst_rect = {
            .x = f_bullet->bullet_pos.x,
            .y = f_bullet->bullet_pos.y,
            .w = 20,
            .h = 20
        };
        r = SDL_RenderCopy(
            game_data->renderer,
            game_data->f_bullet_tex,
            &src_rect, &dst_rect);
    }

    /* display foes */
    for (i = 0; i < game_state->foes_num; i++) {
        int r;
        struct foe *foe = &game_state->foes[i];
        SDL_Rect dst_rect = {
            .x = foe->foe_pos.x,
            .y = foe->foe_pos.y,
            .w = 20,
            .h = 20
        };
        r = SDL_RenderCopy(
            game_data->renderer,
            foe->foe_texture,
            &src_rect, &dst_rect);
    }

    /* display player's bullets */
    {
        int r;
        int i;
        int num = game_state->p_bullets_num;
        for (i = 0; i < num; i++) {
            SDL_Rect dst_rect = {
                .x = game_state->p_bullets[i].x,
                .y = game_state->p_bullets[i].y,
                .w = 20,
                .h = 20
            };
            r = SDL_RenderCopy(
                game_data->renderer,
                game_data->p_bullet_tex,
                &src_rect, &dst_rect);
        }
    }

    /* display player */
    {
        int r;
        struct player *player = &game_state->player;
        SDL_Rect dst_rect = {
            .x = player->p_pos.x,
            .y = player->p_pos.y,
            .w = 20,
            .h = 20
        };
        r = SDL_RenderCopy(
            game_data->renderer,
            player->p_texture,
            &src_rect, &dst_rect);
    }

    SDL_RenderPresent(game_data->renderer);
}


bool f_bullet_inside(struct foe_bullet *bullet)
{
    int x = bullet->bullet_pos.x;
    int y = bullet->bullet_pos.y;
    return (
        (y < height) &&
        (x < width) &&
        (y > -20) &&
        (x > -20)
    );
}


void point_on_line(
        struct point2d p1,
        struct point2d p2, int i, int t,
        struct point2d *r)
{
    struct point2d v1;
    struct point2d v2;
    struct point2d v3;
    int ti = i - t;
    /*
        (
          ( (p1 * ti) +
            (p2 * t)
          ) / i
        )
    */
    p2_mul(p1, ti, &v1);
    p2_mul(p2, t, &v2);
    p2_add(v1, v2, &v3);
    p2_div(v3, i, r);
}


void remove_foe_bullet(struct game_state *game_state, int n)
{
    int num = game_state->f_bullets_num;
    int i, j;
    if (n != (num - 1))
        for (i = n + 1; i < num; i++) {
            j = i - 1;
            struct foe_bullet *fb1 = &game_state->f_bullets[j];
            struct foe_bullet *fb2 = &game_state->f_bullets[i];
            memcpy(fb1, fb2, sizeof(struct foe_bullet));
        }
    game_state->f_bullets_num--;
}


void step_foes_bullets(struct game_state *game_state, int t)
{
    int num = game_state->f_bullets_num;
    int i;

    /* moving foes' bullets */
    for (i = 0; i < num; i++) {
        struct foe_bullet *f_bullet = &game_state->f_bullets[i];
        int dt = t - f_bullet->bullet_birth;

        point_on_line(
                f_bullet->bullet_line[0],
                f_bullet->bullet_line[1],
                6000,  /* 6 seconds to reach the target */
                dt,
                &f_bullet->bullet_pos);
    }

    /* removing foes' bullets that are outside of the screen */
    for (i = 0; i < num; i++) {
        struct foe_bullet *f_bullet = &game_state->f_bullets[i];
        if (!f_bullet_inside(f_bullet)) {
            remove_foe_bullet(game_state, i);
        }
    }
}


void make_foe_anim(struct timeline *tl, int *len, int t)
{
    int t1 = t;
    int t2 = t + 6000 + rand_int(4000);
    switch (rand_int(7))
    {
    case 0:  /* left to right */
      { struct point2d p1 = { .x = -20,             .y = rand_int(height - 20) };
        struct point2d p2 = { .x = rand_int(width), .y = rand_int(height - 20) };
        struct point2d p3 = { .x = width,           .y = rand_int(height - 20) };

        tl[0].type = TIMELINE_TYPE_EVOL;
        tl[0].evol.time1 = t1;
        tl[0].evol.time2 = t2;
        tl[0].evol.qbcurve[0] = p1;
        tl[0].evol.qbcurve[1] = p2;
        tl[0].evol.qbcurve[2] = p3;

        *len = 1;
      } break;
    case 1:  /* right to left */
      { struct point2d p1 = { .x = width,           .y = rand_int(height - 20) };
        struct point2d p2 = { .x = rand_int(width), .y = rand_int(height - 20) };
        struct point2d p3 = { .x = -20,             .y = rand_int(height - 20) };

        tl[0].type = TIMELINE_TYPE_EVOL;
        tl[0].evol.time1 = t1;
        tl[0].evol.time2 = t2;
        tl[0].evol.qbcurve[0] = p1;
        tl[0].evol.qbcurve[1] = p2;
        tl[0].evol.qbcurve[2] = p3;

        *len = 1;
      } break;
    case 2:
    case 3:
    case 4:  /* top to bottom */
      { struct point2d p1 = { .x = rand_int(width - 20), .y = -20 };
        struct point2d p2 = { .x = rand_int(width - 20), .y = rand_int(height - 20) };
        struct point2d p3 = { .x = rand_int(width - 20), .y = height };

        tl[0].type = TIMELINE_TYPE_EVOL;
        tl[0].evol.time1 = t1;
        tl[0].evol.time2 = t2;
        tl[0].evol.qbcurve[0] = p1;
        tl[0].evol.qbcurve[1] = p2;
        tl[0].evol.qbcurve[2] = p3;

        *len = 1;
      } break;
    case 5:
    case 6:  /* top to middle, pause, middle to bottom */
      { int t1 = t;
        int t2 = t + 4000 + rand_int(3000);
        int t3 = t2 + 2000 + rand_int(2000);
        int t4 = t3 + 4000 + rand_int(3000);

        struct point2d p1 = { .x = rand_int(width - 20), .y = -20 };
        struct point2d p2 = { .x = rand_int(width - 20), .y = rand_int(height - 20) };
        struct point2d p3 = { .x = rand_int(width - 20), .y = rand_int(height - 20) };
        struct point2d p4 = { .x = rand_int(width - 20), .y = rand_int(height - 20) };
        struct point2d p5 = { .x = rand_int(width - 20), .y = height };

        tl[0].type = TIMELINE_TYPE_EVOL;
        tl[0].evol.time1 = t1;
        tl[0].evol.time2 = t2;
        tl[0].evol.qbcurve[0] = p1;
        tl[0].evol.qbcurve[1] = p2;
        tl[0].evol.qbcurve[2] = p3;

        tl[1].type = TIMELINE_TYPE_FROM;
        tl[1].from.time = t2;
        tl[1].from.p = p3;

        tl[2].type = TIMELINE_TYPE_EVOL;
        tl[2].evol.time1 = t3;
        tl[2].evol.time2 = t4;
        tl[2].evol.qbcurve[0] = p3;
        tl[2].evol.qbcurve[1] = p4;
        tl[2].evol.qbcurve[2] = p5;

        *len = 3;
      } break;
    }
}


void set_point2d(struct point2d *p, int x, int y)
{
    p->x = x;
    p->y = y;
}

void copy_point2d(struct point2d *r, struct point2d *p)
{
    r->x = p->x;
    r->y = p->y;
}


Uint32 pixel_for_surface(SDL_Surface *surface, struct rgb *rgb)
{
    SDL_PixelFormat *pixel_format = surface->format;
    Uint32 fmt = pixel_format->format;
    SDL_PixelFormat *px_fmt = SDL_AllocFormat(fmt);
    Uint32 pixel = SDL_MapRGB(px_fmt,
        rgb->r,
        rgb->g,
        rgb->b);
    SDL_FreeFormat(px_fmt);
    return pixel;
}


SDL_Texture * make_avatar(SDL_Renderer *renderer, struct rgb *color)
{
    int width = 5;
    int height = 5;
    SDL_Texture *texture;
    Uint32 key;
    Uint32 pixel;
    struct rgb rgb = { .r = 255, .g = 255, .b = 255 };
    int x1, x2, y, n;

    SDL_Surface *surf = SDL_CreateRGBSurface(
        0, width, height,
        32, 0, 0, 0, 0);

    key = pixel_for_surface(surf, &rgb);
    SDL_SetColorKey(surf, 1, key);

    if (color == NULL) {
        rgb.r = 155 + rand_int(100);
        rgb.g = 155 + rand_int(100);
        rgb.b = 155 + rand_int(100);
    } else {
        rgb.r = color->r;
        rgb.g = color->g;
        rgb.b = color->b;
    }
    pixel = pixel_for_surface(surf, &rgb);
    n = 0;
    while (n < 5) {
        for (x1 = 0; x1 < 3; x1++) {
          for (y = 0; y < height; y++) {
            x2 = (width - 1) - x1;
            if (rand_bool() && n < 12) {
                SDL_Rect dst1 = { .x = x1, .y = y, .w = 1, .h = 1 };
                SDL_Rect dst2 = { .x = x2, .y = y, .w = 1, .h = 1 };
                SDL_FillRect(surf, &dst1, pixel);
                SDL_FillRect(surf, &dst2, pixel);
                n++;
            } else {
                SDL_Rect dst1 = { .x = x1, .y = y, .w = 1, .h = 1 };
                SDL_Rect dst2 = { .x = x2, .y = y, .w = 1, .h = 1 };
                SDL_FillRect(surf, &dst1, 0xFFFFFF);
                SDL_FillRect(surf, &dst2, 0xFFFFFF);
            }
          }
        }
    }
    texture = SDL_CreateTextureFromSurface(renderer, surf);
    SDL_FreeSurface(surf);
    return texture;
}


SDL_Texture * texture_of_pattern(
        SDL_Renderer *renderer, char pattern[5][5], struct rgb *color)
{
    int width = 5;
    int height = 5;
    int x, y;
    struct rgb rgb = { .r = 255, .g = 255, .b = 255 };
    Uint32 key;
    Uint32 pixel;
    SDL_Texture *texture;
    SDL_Surface *surf = SDL_CreateRGBSurface(
        0, width, height,
        32, 0, 0, 0, 0);
    key = pixel_for_surface(surf, &rgb);
    SDL_SetColorKey(surf, 1, key);
    pixel = pixel_for_surface(surf, color);
    for (x = 0; x < width; x++) {
        for (y = 0; y < height; y++) {
            SDL_Rect r = { .x = x, .y = y, .w = 1, .h = 1 };
            if (pattern[y][x]) {
                SDL_FillRect(surf, &r, pixel);
            } else {
                SDL_FillRect(surf, &r, 0xFFFFFF);
            }
        }
    }
    texture = SDL_CreateTextureFromSurface(renderer, surf);
    SDL_FreeSurface(surf);
    return texture;
}


void new_foe(SDL_Renderer *renderer, int t, struct foe *foe)
{
    foe->foe_texture = make_avatar(renderer, NULL);
    set_point2d(&foe->foe_pos, rand_int(width - 20), -20);
    make_foe_anim(foe->foe_anim, &foe->foe_anim_len, t);
    foe->foe_last_shot = t;
    foe->foe_shoot_freq = 1600 + rand_int(1800);
}


void new_foes_opt(struct game_state *game_state, struct game_data *game_data, int t)
{
    if (rand_int(100) < 3) {
        int n = game_state->foes_num;
        if (n < MAX_NUM_FOES) {
            new_foe(game_data->renderer, t, &game_state->foes[n]);
            game_state->foes_num++;
        }
    }
}


void remove_foe(struct game_state *game_state, int n)
{
    int i, j;
    if (n != (game_state->foes_num - 1))
        for (i = n + 1; i < game_state->foes_num; i++) {
            j = i - 1;
            struct foe *foe1 = &game_state->foes[j];
            struct foe *foe2 = &game_state->foes[i];
            memcpy(foe1, foe2, sizeof(struct foe));
        }
    game_state->foes_num--;
}


int foe_inside(struct foe *foe, int t)
{
    return !(timeline_finished(foe->foe_anim, foe->foe_anim_len, t));
}


void new_foe_bullet(
        struct game_state *game_state,
        struct point2d *p,
        struct point2d *target, int t)
{
    int i = game_state->f_bullets_num;
    if (i < MAX_F_BULLETS_NUM)
    {
        struct foe_bullet *f_bullet = &game_state->f_bullets[i];
        {
            copy_point2d(&f_bullet->bullet_pos, p);
            copy_point2d(&f_bullet->bullet_line[0], p);
            copy_point2d(&f_bullet->bullet_line[1], target);
            f_bullet->bullet_birth = t;
        }
        game_state->f_bullets_num++;
    }
}


void gun_new_f_bullets(struct game_state *game_state, int t)
{
    int i;
    for (i = 0; i < game_state->foes_num; i++)
    {
        struct foe *foe = &game_state->foes[i];

        if (t - foe->foe_last_shot >= foe->foe_shoot_freq)
        {
            foe->foe_last_shot = t;
            new_foe_bullet(game_state, &foe->foe_pos, &game_state->player.p_pos, t);
        }
    }
}


bool foe_touched(struct game_state *game_state, struct foe *foe)
{
    SDL_Rect foe_rect = {
        .x = foe->foe_pos.x,
        .y = foe->foe_pos.y,
        .w = 20,
        .h = 20
    };

    int i, num = game_state->p_bullets_num;

    for (i = 0; i < num; i++) {
        SDL_Rect bullet_rect = {
            .x = game_state->p_bullets[i].x,
            .y = game_state->p_bullets[i].y,
            .w = 20,
            .h = 20
        };
        if (SDL_HasIntersection(&foe_rect, &bullet_rect)) return true;
    }
    return false;
}


void step_foes(struct game_state *game_state, struct game_data *game_data, int t)
{
    int i;

    new_foes_opt(game_state, game_data, t);
    gun_new_f_bullets(game_state, t);

    /* update foes positions */
    for (i = 0; i < game_state->foes_num; i++) {
        struct foe *foe = &game_state->foes[i];
        timeline_val_at(foe->foe_anim, foe->foe_anim_len, t, &foe->foe_pos);
    }

    /* remove foes which already reached outside of the screen */
    for (i = 0; i < game_state->foes_num; i++) {
        struct foe *foe = &game_state->foes[i];
        if (!foe_inside(foe, t)) {
            missed++;
            SDL_DestroyTexture(foe->foe_texture);
            remove_foe(game_state, i);
        }
    }

    /* remove foes touched by a player's bullet */
    for (i = 0; i < game_state->foes_num; i++) {
        struct foe *foe = &game_state->foes[i];
        if (foe_touched(game_state, foe)) {
            shot++;
            SDL_DestroyTexture(foe->foe_texture);
            remove_foe(game_state, i);
        }
    }
}


bool player_touched(struct game_state *game_state)
{
    struct player *player = &game_state->player;

    SDL_Rect foe_rect = {
        .x = player->p_pos.x,
        .y = player->p_pos.y,
        .w = 20,
        .h = 20
    };

    int i, num = game_state->f_bullets_num;

    for (i = 0; i < num; i++)
    {
        struct foe_bullet *f_bullet = &game_state->f_bullets[i];
        SDL_Rect bullet_rect = {
            .x = f_bullet->bullet_pos.x + 4,
            .y = f_bullet->bullet_pos.y + 4,
            .w = 12,
            .h = 12
        };
        if (SDL_HasIntersection(&foe_rect, &bullet_rect)) return true;
    }
    return false;
}


#define D_LEFT  0x08
#define D_RIGHT 0x04
#define D_UP    0x02
#define D_DOWN  0x01

#define B_1000 0x08
#define B_0100 0x04
#define B_0010 0x02
#define B_0001 0x01

#define B_1010 0x0A
#define B_1001 0x09
#define B_0110 0x06
#define B_0101 0x05

void player_moving(struct player *player)
{
    int p_dir = 0;
    struct point2d *pos = &player->p_pos;

    if (player->p_dir.left ) p_dir |= D_LEFT;
    if (player->p_dir.right) p_dir |= D_RIGHT;
    if (player->p_dir.up   ) p_dir |= D_UP;
    if (player->p_dir.down ) p_dir |= D_DOWN;

    switch (p_dir) {
        case B_1000: pos->x -= 10; break;
        case B_0100: pos->x += 10; break;
        case B_0010: pos->y -= 10; break;
        case B_0001: pos->y += 10; break;

        case B_1010: pos->x -= 7; pos->y -= 7; break;
        case B_1001: pos->x -= 7; pos->y += 7; break;
        case B_0110: pos->x += 7; pos->y -= 7; break;
        case B_0101: pos->x += 7; pos->y += 7; break;
    }

    if (pos->x < 0) pos->x = 0;
    if (pos->y < 0) pos->y = 0;
    if (pos->x > (width - 20))  pos->x = (width - 20);
    if (pos->y > (height - 20)) pos->y = (height - 20);
}


void new_p_bullet(struct game_state *game_state, struct point2d *p)
{
    int i = game_state->p_bullets_num;
    if (i < MAX_P_BULLETS_NUM) {
        game_state->p_bullets[i].x = p->x;
        game_state->p_bullets[i].y = p->y;
        game_state->p_bullets_num++;
    }
}


void remove_p_bullet(struct game_state *game_state, int n)
{
    int i, j;
    int num = game_state->p_bullets_num;

    if (n != (num - 1))
        for (i = n + 1; i < num; i++) {
            j = i - 1;
            struct point2d *pb1 = &game_state->p_bullets[j];
            struct point2d *pb2 = &game_state->p_bullets[i];
            memcpy(pb1, pb2, sizeof(struct point2d));
        }

    game_state->p_bullets_num--;
}


void step_player_bullets(struct game_state *game_state)
{
    int num = game_state->p_bullets_num;
    int i;
    /* moving player's bullets */
    for (i = 0; i < num; i++) {
        struct point2d *pb = &game_state->p_bullets[i];
        pb->y -= 8;
    }
    /* removing player's bullets that are outside of the screen */
    for (i = 0; i < num; i++) {
        struct point2d *pb = &game_state->p_bullets[i];
        if (pb->y < -20)
            remove_p_bullet(game_state, i);
    }
}


void player_shooting(struct game_state *game_state, int t)
{
    struct player *player = &game_state->player;
    if (player->p_shooting &&
        t - player->p_last_shot > player->p_shoot_freq)  /* player shooting */
    {
        new_p_bullet(game_state, &player->p_pos);
        player->p_last_shot = t;
    }
}


void step_player(struct game_state *game_state, int t)
{
    player_moving(&game_state->player);
    player_shooting(game_state, t);
}


void reinit_game(struct game_state *game_state, struct game_data *game_data);


void event_loop(struct game_state *game_state, struct game_data *game_data)
{
    struct player *player = &game_state->player;
    SDL_Event event;
    while (SDL_PollEvent(&event))
    {
        switch (event.type)
        {
            case SDL_QUIT:
                SDL_Quit();
                exit(0);
                break;
            case SDL_KEYDOWN:
                switch (event.key.keysym.sym) {
                    case SDLK_ESCAPE:
                    case SDLK_q:
                        SDL_Quit();
                        exit(0);
                        break;
                    case SDLK_LEFT:
                        player->p_dir.left = true;
                        break;
                    case SDLK_RIGHT:
                        player->p_dir.right = true;
                        break;
                    case SDLK_UP:
                        player->p_dir.up = true;
                        break;
                    case SDLK_DOWN:
                        player->p_dir.down = true;
                        break;
                    case SDLK_z:
                        player->p_shooting = true;
                        break;
                    case SDLK_SPACE:
                        if (game_state->game_over) {
                            game_state->game_over = false;
                            reinit_game(game_state, game_data);
                        } else {
                            timer_toggle_pause(&ptimer);
                            int r = SDL_ShowCursor(timer_is_paused(&ptimer));
                        }
                        break;
                }
                break;
            case SDL_KEYUP:
                switch (event.key.keysym.sym) {
                    case SDLK_LEFT:
                        player->p_dir.left = false;
                        break;
                    case SDLK_RIGHT:
                        player->p_dir.right = false;
                        break;
                    case SDLK_UP:
                        player->p_dir.up = false;
                        break;
                    case SDLK_DOWN:
                        player->p_dir.down = false;
                        break;
                    case SDLK_z:
                        player->p_shooting = false;
                        break;
                }
                break;

            case SDL_JOYBUTTONDOWN:
                if (event.jbutton.which == 0 && event.jbutton.button == 0)
                    player->p_shooting = true;

                if (event.jbutton.which == 0 &&
                    (event.jbutton.button == 2 ||
                     event.jbutton.button == 9))
                    if (game_state->game_over) {
                        game_state->game_over = false;
                        reinit_game(game_state, game_data);
                    } else {
                        timer_toggle_pause(&ptimer);
                        int r = SDL_ShowCursor(timer_is_paused(&ptimer));
                    }
                break;

            case SDL_JOYBUTTONUP:
                if (event.jbutton.which == 0 && event.jbutton.button == 0)
                    player->p_shooting = false;
                break;

            case SDL_JOYHATMOTION:
                if (event.jbutton.which == 0)
                    switch (event.jhat.value)
                    {
                        case SDL_HAT_CENTERED:
                            player->p_dir.left = false;
                            player->p_dir.right = false;
                            player->p_dir.up = false;
                            player->p_dir.down = false;
                            break;
                        case SDL_HAT_UP:
                            player->p_dir.left = false;
                            player->p_dir.right = false;
                            player->p_dir.up = true;
                            player->p_dir.down = false;
                            break;
                        case SDL_HAT_LEFT:
                            player->p_dir.left = true;
                            player->p_dir.right = false;
                            player->p_dir.up = false;
                            player->p_dir.down = false;
                            break;
                        case SDL_HAT_RIGHT:
                            player->p_dir.left = false;
                            player->p_dir.right = true;
                            player->p_dir.up = false;
                            player->p_dir.down = false;
                            break;
                        case SDL_HAT_DOWN:
                            player->p_dir.left = false;
                            player->p_dir.right = false;
                            player->p_dir.up = false;
                            player->p_dir.down = true;
                            break;
                        case SDL_HAT_LEFTUP:
                            player->p_dir.left = true;
                            player->p_dir.right = false;
                            player->p_dir.up = true;
                            player->p_dir.down = false;
                            break;
                        case SDL_HAT_RIGHTUP:
                            player->p_dir.left = false;
                            player->p_dir.right = true;
                            player->p_dir.up = true;
                            player->p_dir.down = false;
                            break;
                        case SDL_HAT_LEFTDOWN:
                            player->p_dir.left = true;
                            player->p_dir.right = false;
                            player->p_dir.up = false;
                            player->p_dir.down = true;
                            break;
                        case SDL_HAT_RIGHTDOWN:
                            player->p_dir.left = false;
                            player->p_dir.right = true;
                            player->p_dir.up = false;
                            player->p_dir.down = true;
                            break;
                    }
                break;
        }
    }
}


void print_score()
{
    printf("# shot: %d\n", shot);
    printf("# missed: %d\n", missed);
    printf("# score: %d\n\n", (shot - missed));
    fflush(stdout);
}


void init_game(struct game_state *game_state, struct game_data *game_data)
{
    struct player *player = &game_state->player;

    struct rgb blue = { .r = 0, .g = 0, .b = 255 };
    struct rgb yellow = { .r = 255, .g = 255, .b = 0 };
    struct rgb green = { .r = 0, .g = 255, .b = 0 };

    init_background();
    init_pausable_timer(&ptimer);

    player->p_pos.x = width / 2 - 10;
    player->p_pos.y = height - 110;
    player->p_last_shot = timer_get_ticks(&ptimer);
    player->p_shoot_freq = 200;
    player->p_shooting = false;
    player->p_dir.left = false;
    player->p_dir.right = false;
    player->p_dir.up = false;
    player->p_dir.down = false;
    player->p_texture = make_avatar(game_data->renderer, &blue);

    game_state->foes_num = 0;
    game_state->p_bullets_num = 0;
    game_state->f_bullets_num = 0;
    game_state->game_over = false;

    /* foes bullet pattern */
    char fb_pattern[5][5] = {
        { 0, 0, 0, 0, 0 },
        { 0, 1, 0, 1, 0 },
        { 0, 0, 0, 0, 0 },
        { 0, 1, 0, 1, 0 },
        { 0, 0, 0, 0, 0 },
    };
    /* player bullet pattern */
    char pb_pattern[5][5] = {
        { 1, 0, 0, 0, 1 },
        { 1, 0, 0, 0, 1 },
        { 0, 0, 0, 0, 0 },
        { 1, 0, 0, 0, 1 },
        { 1, 0, 0, 0, 1 },
    };

    game_data->f_bullet_tex = texture_of_pattern(game_data->renderer, fb_pattern, &yellow);
    game_data->p_bullet_tex = texture_of_pattern(game_data->renderer, pb_pattern, &green);

    /* initialise letters textures */
    {
        int i, n = sizeof(letters) / sizeof(struct letter_pattern);

        for (i = 0; i < n; i++) {
            unsigned char l = letters[i].letter;
            game_data->letters_tex[l] =
                texture_of_pattern(game_data->renderer, letters[i].pattern, &green);
        }
    }
}


void reinit_game(struct game_state *game_state, struct game_data *game_data)
{
    struct player *player = &game_state->player;

    struct rgb blue = { .r = 0, .g = 0, .b = 255 };

    init_background();

    player->p_pos.x = width / 2 - 10;
    player->p_pos.y = height - 110;
    player->p_last_shot = timer_get_ticks(&ptimer);
    SDL_DestroyTexture(player->p_texture);
    player->p_texture = make_avatar(game_data->renderer, &blue);

    game_state->foes_num = 0;
    game_state->p_bullets_num = 0;
    game_state->f_bullets_num = 0;
    game_state->game_over = false;

    shot = 0;
    missed = 0;
}


int main(int argc, char* argv[])
{
    SDL_Window *window;
    SDL_Renderer *renderer;

    struct game_state game_state;
    struct game_data game_data;

    int num_j;

    SDL_Init(
        SDL_INIT_VIDEO |
        SDL_INIT_AUDIO |
        SDL_INIT_TIMER);

    window = SDL_CreateWindow(
        "cshmup-av",
        SDL_WINDOWPOS_UNDEFINED,
        SDL_WINDOWPOS_UNDEFINED,
        width, height,
        SDL_WINDOW_SHOWN
    );

    if (window == NULL) {
        fprintf(stderr, "Could not create window: %s\n", SDL_GetError());
        return 1;
    }

    srand(time(NULL));

    //test_timeline();

    game_data.renderer = SDL_CreateRenderer(window, -1, 0);

    init_game(&game_state, &game_data);

    SDL_InitSubSystem(SDL_INIT_JOYSTICK);

    num_j = SDL_NumJoysticks();
    if (num_j >= 1) {
        SDL_Joystick *joy = SDL_JoystickOpen(0);
    }

    int r = SDL_ShowCursor(false);

    while (1) {
        if (!game_state.game_over && !timer_is_paused(&ptimer)) {
            bool player_is_touched;
            Uint32 t, t2, dt, d;

            t = timer_get_ticks(&ptimer);

            event_loop(&game_state, &game_data);

            /* states steps */
            step_foes(&game_state, &game_data, t);
            step_foes_bullets(&game_state, t);
            step_player(&game_state, t);
            step_player_bullets(&game_state);
            step_background();

            display(&game_state, &game_data);
            player_is_touched = player_touched(&game_state);

            t2 = timer_get_ticks(&ptimer);
            dt = t2 - t;
            d = 30 - dt;
            if (d > 0) SDL_Delay(d);

            if (player_is_touched) {
                game_state.game_over = true;
                print_score();
            }
        } else {
            /* game over */
            event_loop(&game_state, &game_data);
            display(&game_state, &game_data);
            SDL_Delay(100);
        }
    }

    SDL_DestroyWindow(window);
    SDL_Quit();
    return 0;
}

/* vim: set ts=4 sw=4 et: */
