/*
 Copyright (c) 2010 Florent Monnier
 
 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:
 
 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.
 
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
*/

#define GL_GLEXT_PROTOTYPES

#if defined(__APPLE__) && !defined (VMDMESA)
  #include <OpenGL/gl.h>
  #include <OpenGL/glext.h>
#else
  #if defined(USE_GL3_CORE_PROFILE)
    #define GL3_PROTOTYPES 1
    #include <GL3/gl3.h>
  #else
    #include <GL/gl.h>
    #include <GL/glext.h>
  #endif
#endif

/* ==== Simple Example ================================ */

#include <gln.h>
#include <math.h>

#include <GL/glu.h>

/* using the sfml for windowing */

#include <SFML/Graphics.h>
#include <SFML/System.h>
#include <string.h>

void sfProcessEvents(sfRenderWindow* App, gln_matrices *matrices)
{
    sfEvent Event;
    while (sfRenderWindow_GetEvent(App, &Event))
    {
      if (Event.Type == sfEvtClosed ||
         ((Event.Type == sfEvtKeyPressed) && (Event.Key.Code == sfKeyEscape))) {
          sfRenderWindow_Close(App);
      }

      if (Event.Type == sfEvtResized) {
          glViewport(0, 0, Event.Size.Width, Event.Size.Height);
          glMatrixMode(GL_PROJECTION);
          glLoadIdentity();
          gluPerspective(60.f, (float) Event.Size.Width /
                               (float) Event.Size.Height, 0.2f, 60.f);
          glGetFloatv(GL_PROJECTION_MATRIX, matrices->projection);
      }
    }
}

int main()
{
    int width = 800;
    int height = 600;

    gln_matrices matrices;
    gln_drawMeshParams dm_params;

    sfRenderWindow* App;
    sfClock *Clock;
    sfWindowSettings Settings = {24, 8, 0};
    sfVideoMode Mode = {width, height, 32};

    float gl_matrix[16];

    App = sfRenderWindow_Create(
                Mode, "test GLU cylinder",
                sfResize | sfClose, Settings);
    if (!App) return EXIT_FAILURE;

    sfRenderWindow_PreserveOpenGLStates(App, sfTrue);

    double radius = 1.6;
    int slices = 32;
    int stacks = 16;

    radius = 0.2;

    int gen_base           = 0;
    int gen_uv_only        = GLN_GEN_UV;
    int gen_normals_only   = GLN_GEN_NORMALS;
    int gen_normals_and_uv = GLN_GEN_NORMALS | GLN_GEN_UV;

    gln_mesh *my_mesh = glnMakeSphere(radius, slices, stacks, gen_uv_only);

    GLUquadric *q = gluNewQuadric();
    gluQuadricNormals(q, GL_SMOOTH);
    gluQuadricTexture(q, GL_TRUE);

    /* Load an OpenGL texture. */
    GLuint Texture = 0;
    {
        sfImage* Image;
        Image = sfImage_CreateFromFile("./tex3.png");
        if (!Image) return EXIT_FAILURE;

        glGenTextures(1, &Texture);
        glBindTexture(GL_TEXTURE_2D, Texture);

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA,
                     sfImage_GetWidth(Image),
                     sfImage_GetHeight(Image), 0,
                     GL_RGBA, GL_UNSIGNED_BYTE,
                     sfImage_GetPixelsPtr(Image));

        sfImage_Destroy(Image);
    }

    /* Bind our texture */
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, Texture);

    /* enable Z-buffer read and write */
    glEnable(GL_DEPTH_TEST);
    glDepthMask(GL_TRUE);

    /* enable back face culling */
    glFrontFace(GL_CCW);     /* assume a clean model */
    glEnable(GL_CULL_FACE);  /* activate elimination of polygons */
    glCullFace(GL_BACK);     /* remove back side of polygons */

    /* background color */
    glClearColor(0.3, 0.3, 0.35, 1.0);

    /* setup a perspective projection */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(60.f, (float) width / (float) height, 0.2f, 60.f);
    /* let's cheat */
    glGetFloatv(GL_PROJECTION_MATRIX, matrices.projection);

    /* create a clock for measuring the time elapsed */
    Clock = sfClock_Create();

    /* start the main loop */
    while (sfRenderWindow_IsOpened(App))
    {
        float time = sfClock_GetTime(Clock);
        sfProcessEvents(App, &matrices);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        /* Apply some transformations */
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        glTranslated(0.0, 0.0, -5.0);
        glRotatef(time * 20.0f, 1.0f, 0.0f, 0.0f);
        glRotatef(time * 12.0f, 0.0f, 1.0f, 0.0f);
        glRotatef(time * 36.0f, 0.0f, 0.0f, 1.0f);

        glGetFloatv(GL_MODELVIEW_MATRIX, matrices.modelview);

        float v = (cosf(time) + 1.0) / 2.0;
        dm_params.color[0] = v;        /* r */
        dm_params.color[1] = 1.0 - v;  /* g */
        dm_params.color[2] = 0.0;      /* b */

        float a = time * 1.1f;
        float b = time * 0.9f;
        /* set the light direction */
        dm_params.light_dir[0] = cos(a) * cos(b);  /* x */
        dm_params.light_dir[1] = cos(a) * sin(b);  /* y */
        dm_params.light_dir[2] = sin(a);           /* z */

        double base = 1.2;
	double top  = 0.0;
	double height = 2.0;
	int slices = 24;
	int stacks = 3;
        gluCylinder(q, base, top, height, slices, stacks);
        //glnDrawMesh(my_mesh, &matrices, &dm_params);
        gluSphere(q, radius, slices, stacks);

        sfRenderWindow_Display(App);
    }

    /* cleanup gln-lib resources */
    glnDeleteMesh(my_mesh);

    /* Don't forget to destroy our texture */
    glDeleteTextures(1, &Texture);

    /* cleanup windowing resources */
    sfClock_Destroy(Clock);
    sfRenderWindow_Destroy(App);

    return EXIT_SUCCESS;
}

