(* Simple 2D physics *)
(* Copyright (C) 2024 Florent Monnier *)
(* Permission is granted to anyone to use this software for any purpose,
   including commercial applications, and to modify it and redistribute it freely.
 *)

type point = float * float
type vector = float * float

type segment = point * point

type 'a circle = {
  cx: float;  (* center (cx, cy) *)
  cy: float;
  r: float;  (* radius *)
  vx: float;  (* inertia vector (vx, vy) *)
  vy: float;
  static: bool;  (* static circles don't move *)
  custom : 'a;
}


(* Function to apply gravity to a circle *)
let apply_gravity circle (gx, gy) dt =
  { circle with
    vx = circle.vx +. gx *. dt;
    vy = circle.vy +. gy *. dt;
  }
;;


(* Function to normalize a vector (x, y) *)
let normalize_vector (x, y) =
  let magnitude = sqrt (x *. x +. y *. y) in
  if magnitude = 0. then
    (0., 0.)  (* Handle the zero vector case to avoid division by zero *)
  else
    (x /. magnitude, y /. magnitude)


(* Function to handle collision between a circle and a segment *)
let collision_circle_segment circle ((p1x, p1y), (p2x, p2y)) restitution =
  if circle.static then circle else

  let cx, cy, r = circle.cx, circle.cy, circle.r in
  let vx, vy = circle.vx, circle.vy in

  (* Calculate the minimum and maximum x and y values of the segment *)
  let min_x = min p1x p2x in
  let min_y = min p1y p2y in
  let max_x = max p1x p2x in
  let max_y = max p1y p2y in

  (* Quick check if the circle can be in collision *)
  if (cx +. r <= min_x) ||
     (cx -. r >= max_x) ||
     (cy +. r <= min_y) ||
     (cy -. r >= max_y)
  then circle
  else
    (* Calculate the direction vector of the segment *)
    let dx = p2x -. p1x in
    let dy = p2y -. p1y in

    (* Calculate the vector from the start of the segment to the center of the circle *)
    let fx = cx -. p1x in
    let fy = cy -. p1y in

    (* Calculate the parameter t for the projection of the circle's center on the segment *)
    let t = (fx *. dx +. fy *. dy) /. (dx *. dx +. dy *. dy) in

    (* Calculate the point of projection on the segment *)
    let projection_x, projection_y =
      if t < 0.0 then (p1x, p1y)
      else if t > 1.0 then (p2x, p2y)
      else (p1x +. t *. dx, p1y +. t *. dy)
    in

    (* Calculate the distance between the center of the circle and the projection point *)
    let distance = sqrt ((cx -. projection_x) ** 2.0 +. (cy -. projection_y) ** 2.0) in

    (* Check for collision *)
    if distance > r then circle
    else
      (* Collision detected, calculate the new velocity vector *)
      let normal_x = (cx -. projection_x) /. distance in
      let normal_y = (cy -. projection_y) /. distance in
      let dot_product = vx *. normal_x +. vy *. normal_y in

      let penetration_depth = r -. distance in

      { circle with

        (* Update the velocity vector by inverting the normal component and applying restitution *)
        vx = vx -. (1. +. restitution) *. dot_product *. normal_x;
        vy = vy -. (1. +. restitution) *. dot_product *. normal_y;
       
        (* Separate the circle from the segment to avoid consecutive collisions *)
        cx = cx +. normal_x *. penetration_depth;
        cy = cy +. normal_y *. penetration_depth;
      }
;;


(* Function to handle collision between two circles *)
let collision_circle_circle circle1 circle2 restitution =
  let dx = circle2.cx -. circle1.cx in
  let dy = circle2.cy -. circle1.cy in
  let distance = sqrt (dx *. dx +. dy *. dy) in
  let overlap = circle1.r +. circle2.r -. distance in

  if overlap <= 0.0 then (circle1, circle2)
  else
    let normal_x, normal_y = normalize_vector (dx, dy) in
    let relative_velocity_x = circle1.vx -. circle2.vx in
    let relative_velocity_y = circle1.vy -. circle2.vy in
    let dot_product = relative_velocity_x *. normal_x +. relative_velocity_y *. normal_y in

    if dot_product <= 0.0 then (circle1, circle2)
    else
      (* Resolve the collision by adjusting the velocities *)
      let impulse = (1. +. restitution) *. dot_product /. (circle1.r +. circle2.r) in

      (* Separate the circles to avoid consecutive collisions *)
      let separation = overlap /. 2.0 in

      let circle1 =
        if circle1.static then circle1 else
          { circle1 with
            vx = circle1.vx -. impulse *. normal_x;
            vy = circle1.vy -. impulse *. normal_y;

            cx = circle1.cx -. normal_x *. separation;
            cy = circle1.cy -. normal_y *. separation; }
      in
      let circle2 =
        if circle2.static then circle2 else
          { circle2 with
            vx = circle2.vx +. impulse *. normal_x;
            vy = circle2.vy +. impulse *. normal_y;

            cx = circle2.cx +. normal_x *. separation;
            cy = circle2.cy +. normal_y *. separation; }
      in

      (circle1, circle2)
;;


(* Function to update the position of multiple circles and handle collisions *)
let update_circles circles segments gravity dt restitution =
  let circles =
    List.map (fun circle ->
      if circle.static then circle else
      begin
        (* Apply gravity to the circle's velocity *)
        let circle = apply_gravity circle gravity dt in
       
        (* Update the circle's position based on its velocity *)
        let circle =
          { circle with
            cx = circle.cx +. circle.vx *. dt;
            cy = circle.cy +. circle.vy *. dt;
          }
        in
       
        (* Check for collisions with each segment *)
        List.fold_left (fun circle segment ->
          collision_circle_segment circle segment restitution
        ) circle segments
      end
    ) circles
  in

  (* Check for collisions between circles *)
  let rec check_circle_collisions circles acc =
    match circles with
    | [] -> (List.rev acc)
    | circle1 :: rest ->
        let circle1, rest =
          List.fold_left (fun (circle1, acc) circle2 ->
            let circle1, circle2 = collision_circle_circle circle1 circle2 restitution in
            (circle1, circle2::acc)
          ) (circle1, []) rest
        in
        check_circle_collisions rest (circle1::acc)
  in
  check_circle_collisions circles []
;;

